package com.agilex.healthcare.directscheduling.dataservice;

import com.agilex.healthcare.directscheduling.dao.DAOConstants;
import com.agilex.healthcare.directscheduling.dao.PatientRelationshipDaoImpl;
import com.agilex.healthcare.directscheduling.domain.DirectSchedulingFacilityProvider;
import com.agilex.healthcare.directscheduling.domain.DirectSchedulingFacilityProviders;
import com.agilex.healthcare.directscheduling.domain.FacilityMember;
import com.agilex.healthcare.directscheduling.domain.FacilityMemberClinic;
import com.agilex.healthcare.directscheduling.domain.FacilityMemberTeam;
import com.agilex.healthcare.directscheduling.domain.facility.DSFacilities;
import com.agilex.healthcare.directscheduling.domain.facility.DSFacility;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientIdentifier;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientProvider;
import com.agilex.healthcare.mobilehealthplatform.domain.PatientProviders;
import com.agilex.healthcare.veteranappointment.dataservice.ProviderDataService;
import com.agilex.healthcare.veteranappointment.domain.PatientProviderWithFacility;
import gov.va.vamf.scheduling.direct.datalayer.eligibility.ClinicalServiceService;
import gov.va.vamf.scheduling.direct.domain.ClinicalService;
import gov.va.vamf.scheduling.direct.domain.StopCode;
import gov.va.vamf.scheduling.direct.domain.StopCodes;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import static org.apache.commons.lang.StringUtils.equalsIgnoreCase;
import static org.apache.commons.lang.StringUtils.isEmpty;
import static org.apache.commons.lang.StringUtils.isNotEmpty;

@Service
@Scope(BeanDefinition.SCOPE_SINGLETON)
public class DSFacilityProviderDataService extends DSDataService {

    @Resource(name = "patientRelationshipDaoClinicImpl")
    private PatientRelationshipDaoImpl patientRelationshipDaoClinicImpl;

    @Resource(name = "patientRelationshipDaoTeamImpl")
    private PatientRelationshipDaoImpl patientRelationshipDaoTeamImpl;

    @Resource
    ProviderDataService providerDataService;

    @Resource
    ClinicalServiceService clinicalServiceService;

    public DSFacilityProviderDataService() {
    }

    public DSFacilities getAllFacilityProvidersForPatient(PatientIdentifier patientIdentifer) {

        PatientIdentifier icnIdentifier = getIcnIdentifiers(patientIdentifer);
        if (icnIdentifier == null || !"ICN".equalsIgnoreCase(icnIdentifier.getAssigningAuthority())) {
            return (null);
        }

        DSFacilities dsFacilities = new DSFacilities();

        PatientProviders providers = providerDataService.fetchPrimaryCareProviders(icnIdentifier, null, false);
        providers.addAll(providerDataService.fetchMentalHealthProviders(icnIdentifier, null, false));
        loadVARFacilityMembersIntoDSFacilities(providers, dsFacilities);

        return (dsFacilities.getFacilityList().size() == 0 ? null : dsFacilities);
    }

    public DirectSchedulingFacilityProviders getPactFacilityMembersForPatient(PatientIdentifier patientIdentifier, String siteCode,
                                                                              boolean canSchedule, String stopCodePairs) {

        PatientIdentifier icnIdentifier = getIcnIdentifiers(patientIdentifier);
        if (icnIdentifier == null || !"ICN".equalsIgnoreCase(icnIdentifier.getAssigningAuthority())) {
            return (null);
        }

        DirectSchedulingFacilityProviders dsfps = null;
        Map<String, Object> inputParamsMap = new HashMap<>();
        inputParamsMap.put(DAOConstants.PATIENT_ICN, icnIdentifier.getUniqueId());
        inputParamsMap.put(DAOConstants.STOP_CODE_PAIRS, stopCodePairs);

        Collection<FacilityMember> facilityTeam = patientRelationshipDaoTeamImpl.getPatientData(inputParamsMap);
        facilityTeam = filterFacilityMembers(siteCode, facilityTeam);

        Collection<FacilityMember> facilityClinics = patientRelationshipDaoClinicImpl.getPatientData(inputParamsMap);
        facilityClinics = filterFacilityMembers(siteCode, facilityClinics);

        dsfps = getPactFacilityMembers(facilityTeam, facilityClinics, canSchedule);

        return dsfps == null ? dsfps : mergeFacilityProvidersByFacility(dsfps);
    }

    public Collection<FacilityMemberTeam> getTeamMembers(PatientIdentifier patientIdentifier, String siteCode, String clinicalServiceId, TeamListCodes teamListCodes) {

        Collection<FacilityMemberTeam> teamMembers = new ArrayList<>();

        final ClinicalService clinicalService = clinicalServiceService.fetchById(clinicalServiceId);
        if (clinicalService != null) {
            DirectSchedulingFacilityProviders dsfps = getPactFacilityMembersForPatient(patientIdentifier, siteCode, false, ClinicalService.getStopCodesAsString(clinicalService.getStopCodes()));

            if (dsfps != null) {
                dsfps = mergeFacilityProvidersByFacility(dsfps);

                Iterator<DirectSchedulingFacilityProvider> directSchedulingFacilityProviderIterator = dsfps.getFacilityProviderList().iterator();

                while (directSchedulingFacilityProviderIterator.hasNext()) {
                    DirectSchedulingFacilityProvider directSchedulingFacilityProvider = directSchedulingFacilityProviderIterator.next();

                    switch (teamListCodes) {
                        case TEAM_LIST:
                            teamMembers.addAll(directSchedulingFacilityProvider.getTeamList());
                            break;
                        case PROVIDERS:
                            teamMembers.addAll(directSchedulingFacilityProvider.getPrimaryList());
                            break;
                        case PACT_TEAM:
                            teamMembers.addAll(directSchedulingFacilityProvider.getPrimaryList());
                            teamMembers.addAll(directSchedulingFacilityProvider.getTeamList());
                            break;
                        default:
                    }
                }
            }
        }

        return teamMembers;
    }

    public DirectSchedulingFacilityProviders mergeFacilityProvidersByFacility(DirectSchedulingFacilityProviders dsfps) {

        Map<String, DirectSchedulingFacilityProvider> resultMap = new HashMap<String, DirectSchedulingFacilityProvider>();
        populateResultMapWithPrimaryFacility(resultMap, dsfps);
        mergeResultMapWithCbocFacility(resultMap, dsfps);

        DirectSchedulingFacilityProviders mergedProviders = new DirectSchedulingFacilityProviders();
        List<DirectSchedulingFacilityProvider> facilityProviderList = new ArrayList<DirectSchedulingFacilityProvider>();

        Iterator<String> iter = resultMap.keySet().iterator();
        while(iter.hasNext()) {
            String facilityId = iter.next();
            facilityProviderList.add(resultMap.get(facilityId));
        }

        mergedProviders.setFacilityProviderList(facilityProviderList);
        return mergedProviders;
    }

    private void mergeResultMapWithCbocFacility(Map<String, DirectSchedulingFacilityProvider> resultMap,
                                                DirectSchedulingFacilityProviders dsfps) {

        List<DirectSchedulingFacilityProvider> providers = dsfps.getFacilityProviderList();
        for(DirectSchedulingFacilityProvider provider : providers) {
            String facilityId = provider.getFacilityId();
            String institutionCode = provider.getInstitutionCode();

            if(! facilityId.equals(institutionCode)) {
                DirectSchedulingFacilityProvider mergedProvider = resultMap.get(facilityId);

                if(mergedProvider == null) {
                    resultMap.put(facilityId, provider);
                } else {
                    mergedProvider.getClinicList().addAll(provider.getClinicList());
                    mergedProvider.getPrimaryList().addAll(provider.getPrimaryList());
                    mergedProvider.getTeamList().addAll(provider.getTeamList());
                }
            }
        }
    }

    private void populateResultMapWithPrimaryFacility(Map<String, DirectSchedulingFacilityProvider> resultMap,
                                                      DirectSchedulingFacilityProviders dsfps) {
        List<DirectSchedulingFacilityProvider> providers = dsfps.getFacilityProviderList();
        for(DirectSchedulingFacilityProvider provider : providers) {
            String facilityId = provider.getFacilityId();
            String institutionCode = provider.getInstitutionCode();

            if(facilityId.equals(institutionCode)) {
                resultMap.put(facilityId, provider);
            }
        }
    }

    private DirectSchedulingFacilityProviders getPactFacilityMembers(
            Collection<FacilityMember> facilityTeam,
            Collection<FacilityMember> facilityClinics,
            boolean canSchedule
    ) {
        Map<String, Collection<FacilityMemberTeam>> allPrimaryMembers = filterFacilityByPossiblePrimary(facilityTeam, "Y");
        if (allPrimaryMembers == null) {
            return (null);
        }

        Map<String, Collection<FacilityMemberTeam>> allTeamMembers = filterFacilityByTeamMember(facilityTeam);

        Collection<FacilityMemberClinic> validClinics = filterFacilityByInstitutionName(facilityClinics, canSchedule);
        if (validClinics == null) {
            return (null);
        }

        DirectSchedulingFacilityProviders providers = new DirectSchedulingFacilityProviders();
        for (FacilityMemberClinic ftc : validClinics) {

            DirectSchedulingFacilityProvider facilityProviderClinic = getFacilityProviderByFacilityId(providers, ftc.getFacilityId(),
                ftc.getInstitutionCode(), ftc.getInstitutionName());

            Collection<FacilityMemberClinic> clinicList = facilityProviderClinic.getClinicList();
            if (clinicList == null) {
                clinicList = new HashSet<FacilityMemberClinic>();
                facilityProviderClinic.setClinicList(clinicList);
            }
            facilityProviderClinic.getClinicList().add(ftc);

            Collection<FacilityMemberTeam> primaryList = facilityProviderClinic.getPrimaryList();
            if (primaryList == null) {
                primaryList = new HashSet<FacilityMemberTeam>();
                facilityProviderClinic.setPrimaryList(primaryList);
            }
            if (allPrimaryMembers.get(ftc.getFacilityId()) != null) {
                facilityProviderClinic.getPrimaryList().addAll(allPrimaryMembers.get(ftc.getFacilityId()));
            }

            Collection<FacilityMemberTeam> teamList = facilityProviderClinic.getTeamList();
            if (teamList == null) {
                teamList = new HashSet<FacilityMemberTeam>();
                facilityProviderClinic.setTeamList(teamList);
            }
            if (allTeamMembers != null && allTeamMembers.get(ftc.getFacilityId()) != null) {
                facilityProviderClinic.getTeamList().addAll(allTeamMembers.get(ftc.getFacilityId()));
            }
        }

        return (providers);
    }

    private DirectSchedulingFacilityProvider getFacilityProviderByFacilityId(DirectSchedulingFacilityProviders providers,
                                                                             String faciltyId, String institutionCode,
                                                                             String institutionName) {

        List<DirectSchedulingFacilityProvider> facilityProviderCollection = providers.getFacilityProviderList();

        if (facilityProviderCollection == null) {
            facilityProviderCollection = new ArrayList<DirectSchedulingFacilityProvider>();
            providers.setFacilityProviderList(facilityProviderCollection);
        }

        Iterator<DirectSchedulingFacilityProvider> it = facilityProviderCollection.iterator();

        while (it.hasNext()) {
            DirectSchedulingFacilityProvider dsfp = it.next();
            if (dsfp.getFacilityId().equalsIgnoreCase(faciltyId) && dsfp.getInstitutionCode().equalsIgnoreCase(institutionCode)) {
                return (dsfp);
            }
        }

        DirectSchedulingFacilityProvider newFacility = new DirectSchedulingFacilityProvider();
        newFacility.setFacilityId(faciltyId);
        newFacility.setInstitutionCode(institutionCode);
        newFacility.setInstitutionName(institutionName);
        facilityProviderCollection.add(newFacility);
        return (newFacility);
    }

    private Collection<FacilityMemberClinic> filterFacilityByInstitutionName(Collection<? extends FacilityMember> facilityTeamClinics,
                                                                             boolean canSchedule) {
        Collection<FacilityMemberClinic> clinicCollection = new ArrayList<FacilityMemberClinic>();
        Iterator<? extends FacilityMember> teamIt = facilityTeamClinics.iterator();

        while (teamIt.hasNext()) {
            FacilityMemberClinic ftd = (FacilityMemberClinic) teamIt.next();

            if ((canSchedule && isNotEmpty(ftd.getSchedulingFlag()) && equalsIgnoreCase(ftd.getSchedulingFlag(), "Y")) ||
                (!canSchedule && isNotEmpty(ftd.getInstitutionName()) && isNotEmpty(ftd.getInstitutionCode())) )
            {
                if (ftd.getFriendlyLocationName() != null && isNotEmpty(ftd.getFriendlyLocationName())) {
                    ftd.setPhysicalLocation(ftd.getFriendlyLocationName());
                } else {
                    ftd.setPhysicalLocation(ftd.getLocationName());
                }
                clinicCollection.add(ftd);
            }
        }
        return (clinicCollection.size() == 0 ? null : clinicCollection);
    }

    private Map<String, Collection<FacilityMemberTeam>> filterFacilityByTeamMember(Collection<? extends FacilityMember> facilityMemberCollection) {
        Map<String, Collection<FacilityMemberTeam>> doctorCollectionByFacility = new HashMap<String, Collection<FacilityMemberTeam>>();

        Iterator<? extends FacilityMember> teamIt = facilityMemberCollection.iterator();

        while (teamIt.hasNext()) {
            FacilityMemberTeam ftd = (FacilityMemberTeam) teamIt.next();
            if (isEmpty(ftd.getPossiblePrimary()) || ftd.getPossiblePrimary().equalsIgnoreCase("N")) {
                if (!doctorCollectionByFacility.containsKey(ftd.getFacilityId())) {
                    doctorCollectionByFacility.put(ftd.getFacilityId(), new ArrayList<FacilityMemberTeam>());
                }
                doctorCollectionByFacility.get(ftd.getFacilityId()).add(ftd);
            }
        }
        return (doctorCollectionByFacility.size() == 0 ? null : doctorCollectionByFacility);
    }

    private Map<String, Collection<FacilityMemberTeam>> filterFacilityByPossiblePrimary(Collection<? extends FacilityMember> facilityMemberCollection, String possiblePrimaryValue) {
        Map<String, Collection<FacilityMemberTeam>> doctorCollectionByFacility = new HashMap<String, Collection<FacilityMemberTeam>>();

        Iterator<? extends FacilityMember> teamIt = facilityMemberCollection.iterator();

        while (teamIt.hasNext()) {
            FacilityMemberTeam ftd = (FacilityMemberTeam) teamIt.next();
            if (equalsIgnoreCase(ftd.getPossiblePrimary(), possiblePrimaryValue)) {
                if (!doctorCollectionByFacility.containsKey(ftd.getFacilityId())) {
                    doctorCollectionByFacility.put(ftd.getFacilityId(), new ArrayList<FacilityMemberTeam>());
                }
                doctorCollectionByFacility.get(ftd.getFacilityId()).add(ftd);
            }
        }
        return (doctorCollectionByFacility.size() == 0 ? null : doctorCollectionByFacility);
    }

    private DSFacility findFacilityByFacilityCode(String facilityCode, Collection<DSFacility> facilityList) {

        if (facilityList == null || facilityList.size() == 0) {
            return (null);
        }

        for (DSFacility facility : facilityList) {
            if (facility.getFacilityCode() != null && facility.getFacilityCode().equalsIgnoreCase(facilityCode)) {
                return (facility);
            }
        }

        return (null);
    }

    private void loadVARFacilityMembersIntoDSFacilities(PatientProviders providers, DSFacilities dsFacilities) {

        if (providers == null || providers.size() == 0) {
            return;
        }

        List<DSFacility> dsFacilityCollection = dsFacilities.getFacilityList();
        for (PatientProvider provider : providers) {
            PatientProviderWithFacility providerWithFacility = (PatientProviderWithFacility) provider;
            String fmpFacilityCode = providerWithFacility.getFacilityCode();
            String fmpFacilityName = providerWithFacility.getFacilityName();
            String fmpPatientICN = providerWithFacility.getPatientId();

            DSFacility tempFacility = findFacilityByFacilityCode(fmpFacilityCode, dsFacilityCollection);
            if (tempFacility == null) {

                tempFacility = new DSFacility();
                tempFacility.setFacilityCode(fmpFacilityCode);
                tempFacility.setFacilityName(fmpFacilityName);
                tempFacility.setPatientICN(fmpPatientICN);

                dsFacilityCollection.add(tempFacility);
            }
        }
    }

    public void setPatientRelationshipDaoClinicImpl(PatientRelationshipDaoImpl patientRelationshipDaoImpl) {
        this.patientRelationshipDaoClinicImpl = patientRelationshipDaoImpl;
    }

    public void setPatientRelationshipDaoTeamImpl(PatientRelationshipDaoImpl patientRelationshipDaoImpl) {
        this.patientRelationshipDaoTeamImpl = patientRelationshipDaoImpl;
    }

    public StopCodes createPrimaryStopCodeFromString(String clinicalServiceId) {
        StopCodes stopCodes = new StopCodes();
        StopCode stopCode = new StopCode();
        stopCode.setPrimary(clinicalServiceId);
        stopCodes.add(stopCode);

        return stopCodes;
    }

    public StopCodes createStopCodesFromPrimary(String clinicalServiceId) {
        ClinicalService clinicalServices = clinicalServiceService.fetchById(clinicalServiceId);

        if (clinicalServices != null) {
            return clinicalServices.getStopCodes();
        }

        StopCodes stopCodes = new StopCodes();
        StopCode stopCode = new StopCode();
        stopCodes.add(stopCode);

        return stopCodes;
    }

    public PatientProviders getProvidersForPatient(PatientIdentifier patientIdentifer) {

        PatientIdentifier icnIdentifier = getIcnIdentifiers(patientIdentifer);
        if (icnIdentifier == null || !"ICN".equalsIgnoreCase(icnIdentifier.getAssigningAuthority())) {
            return (null);
        }

        PatientProviders patientProviders = new PatientProviders();
        //patientProviders = providerDataService.fetchMentalHealthProviders(icnIdentifier, null, false);
        patientProviders = providerDataService.fetchPrimaryCareProviders(icnIdentifier, null, false);

        return patientProviders;
    }

    private Collection<FacilityMember> filterFacilityMembers(String siteCode, Collection<FacilityMember> varFacilityMembers) {
        Collection<FacilityMember> facilityMembersFiltered = new ArrayList<>(varFacilityMembers.size());

        for (final FacilityMember facilityMember: varFacilityMembers) {
            if (facilityMember.getFacilityId().equals(siteCode)) {
                facilityMembersFiltered.add(facilityMember);
            }
        }

        return facilityMembersFiltered;
    }
}
